﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;

namespace WorldLab.GeoDataLib
{
    ///    r7 — — — — — — — — —
    ///    r6 — — — — — — — — —
    ///    r5 — — — — — — — — —
    /// Y  r4 — — — — — — — — —
    ///    r3 — — — — — — — — —
    ///    r2 — — — — — — — — —
    ///    r1 — — — — — — — — —
    ///    r0 — — — — — — — — —
    ///    O  c0 c1 c2 c3 c4 c5 c6 c7 c8
    ///                   X
    public class Grid
    {
        private int rowCounts;/// row counts
        private int colCounts;/// column counts
        private double originalPosX;/// original point X
        private double originalPosY;/// original point Y
        private double size;/// size
        private double noDataValue;/// noData value
        private double[] values;

        public int RowCounts/// row counts
        {
            get
            {
                return rowCounts;
            }
            set
            {
                rowCounts = value;
            }
        }
        public int ColCounts/// column counts
        {
            get
            {
                return colCounts;
            }
            set
            {
                colCounts = value;
            }
        }
        public double OriginalPosX/// original point X
        {
            get
            {
                return originalPosX;
            }
            set
            {
                originalPosX = value;
            }
        }
        public double OriginalPosY/// original point Y
        {
            get
            {
                return originalPosY;
            }
            set
            {
                originalPosY = value;
            }
        }
        public double Size/// size
        {
            get
            {
                return size;
            }
            set
            {
                size = value;
            }
        }
        public double NoDataValue/// noData value
        {
            get
            {
                return noDataValue;
            }
            set
            {
                noDataValue = value;
            }
        }
        public double[] Values
        {
            get
            {
                return values;
            }
            set
            {
                values = value;
            }
        }

        private bool Init(int rows, int cols)
        {
            int nSize = rows * cols;
            if (nSize <= 0)
            {
                values = null;
                return false;
            }
            values = new double[nSize];

            return true;
        }

        private void SetData(double[] vals)
        {
            values = (double[])vals.Clone();
        }

        public Grid()
        {
            rowCounts = 0;
            colCounts = 0;
            originalPosX = double.NaN;
            originalPosY = double.NaN;
            size = double.NaN;
            noDataValue = double.NaN;
            Init(0, 0);
        }

        public Grid(int rows, int cols, double x, double y, 
            double siz, double noVal)
		{
            rowCounts = rows;
            colCounts = cols;
            originalPosX = x;
            originalPosY = y;
            size = siz;
            noDataValue = noVal;
            Init(rows, cols);
		}

        public Grid(int rows, int cols, double x, double y, 
            double siz, double noVal, double[] vals, int valCounts)
        {
            rowCounts = rows;
            colCounts = cols;
            originalPosX = x;
            originalPosY = y;
            size = siz;
            noDataValue = noVal;
            if (Init(rows, cols) && rows * cols == valCounts)
            {
                SetData(vals);
            }
        }

        public Grid(double[,] values)
        {
            rowCounts = values.GetLength(0);
            colCounts = values.GetLength(1);
            originalPosX = double.NaN;
            originalPosY = double.NaN;
            size = double.NaN;
            noDataValue = double.NaN;
            double[] data = new double[rowCounts * colCounts];
            int k = 0;
            for (int i = 0; i < rowCounts; ++i)
            {
                for (int j = 0; j < colCounts; ++j)
                {
                    data[k++] = values[i, j];
                }
            }
            Init(rowCounts, colCounts);
            SetData(data);
        }

        public Grid(Grid source)
        {
            rowCounts = source.RowCounts;
            colCounts = source.ColCounts;
            originalPosX = source.OriginalPosX;
            originalPosY = source.OriginalPosY;
            size = source.Size;
            noDataValue = source.NoDataValue;
            Init(rowCounts, colCounts);
            SetData(source.values);
        }

        public void Dispose()
        {
            rowCounts = 0;
            colCounts = 0;
            originalPosX = double.NaN;
            originalPosY = double.NaN;
            size = double.NaN;
            noDataValue = double.NaN;
            values = null;
        }

        public double GetValue(int row, int col) 
		{
            if (row < 0 || row >= rowCounts || col < 0 || col >= colCounts)
            {
                return double.NaN;
            }
            else
            {
                return values[row * colCounts + col];
            }
		}

        public bool SetValue(int row, int col, double val)
        {
            if (row < 0 || row >= rowCounts || col < 0 || col >= colCounts)
            {
                return false;
            }
            else
            {
                values[row * colCounts + col] = val;
                return true;
            }
        }

        /// ncols         228
        /// nrows         263
        /// xllcorner     384250.0838
        /// yllcorner     2606582.813
        /// cellsize      2000
        /// NODATA_value  -9999
        private static string[] GetInfoFromTxt(string fileName)
        {
            string[] gridInfoArr = new string[6];

            FileStream fs = new FileStream(fileName, FileMode.Open, FileAccess.Read);
            StreamReader sr = new StreamReader(fs);

            string lineText = "";
            string[] tokens = null;
            int lineIndex = 0;
            while ((lineText = sr.ReadLine()) != null && lineIndex < 6)
            {
                lineText = lineText.Trim();
                tokens = lineText.Split(' ');
                if (tokens[tokens.Length - 1] == null)
                {
                    gridInfoArr = null;
                    break;
                }
                gridInfoArr[lineIndex] = tokens[tokens.Length - 1];
                lineIndex++;
            }

            sr.Close();
            sr.Dispose();
            sr = null;
            fs.Close();
            fs.Dispose();
            fs = null;

            return gridInfoArr;
        }

        private static List<double> GetValuesFromTxt(string fileName)
        {
            List<double> valList = new List<double>();
            FileStream fs = new FileStream(fileName, FileMode.Open, FileAccess.Read);
            StreamReader sr = new StreamReader(fs);
            for (int i = 0; i < 6; i++)
            {
                sr.ReadLine();
            }
            string lineText = "";
            string[] tokens = null;
            double readValue;
            while ((lineText = sr.ReadLine()) != null)
            {
                lineText = lineText.Trim();
                tokens = lineText.Split(' ');
                for (int i = 0; i < tokens.Length; ++i)
                {
                    if (double.TryParse(tokens[i], out readValue))
                    {
                        valList.Add(readValue);
                    }
                    else
                    {
                        valList.Clear();
                        valList = null;
                        break;
                    }
                }
                if (valList == null)
                {
                    break;
                }
            }

            sr.Close();
            sr.Dispose();
            sr = null;
            fs.Close();
            fs.Dispose();
            fs = null;

            return valList;
        }

        public static Grid GetFromTxt(string fileName)
        {
            string[] gridInfoArr = GetInfoFromTxt(fileName);
            int rows, cols;
            double x, y, siz, nodata;
            if (gridInfoArr == null)
            {
                return null;
            }
            if (!int.TryParse(gridInfoArr[0], out cols)     ||
                !int.TryParse(gridInfoArr[1], out rows)     ||
                !double.TryParse(gridInfoArr[2], out x)     ||
                !double.TryParse(gridInfoArr[3], out y)     ||
                !double.TryParse(gridInfoArr[4], out siz)   ||
                !double.TryParse(gridInfoArr[5], out nodata) )
            {
                gridInfoArr = null;
                return null;
            }
            if (rows <= 0 || cols <= 0 || siz <= 0)
            {
                return null;
            }

            List<double> valList = GetValuesFromTxt(fileName);
            if (valList == null)
            {
                gridInfoArr = null;
                return null;
            }
            if (valList.Count != rows * cols)
            {
                gridInfoArr = null;
                valList.Clear();
                valList = null;
                return null;
            }

            Grid grid = new Grid(rows, cols, x, y, siz,
                nodata, valList.ToArray(), rows * cols);
            valList.Clear();
            valList = null;
            return grid;
        }
    }
}
